/**
 *******************************************************************************
 * @file    systick.c
 *
 * @brief   System initialization file.
 *
 * @author  DP
 *
 * @version V1.0.1
 *
 * @date    2024-11-11
 *
 *******************************************************************************/

/*
    Copyright (c) 2024, DP Microelectronics Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

/* Files include */
#include "systick.h"
#include "dpm32m0xx.h"
#include "dpm32m036_libopt.h"

/**
  * @addtogroup DPM32M036_LibSamples
  * @{
  */

/**
  * @addtogroup Templates
  * @{
  */

/* Private typedef --------------------------------------------------------------*/

/* Private define ---------------------------------------------------------------*/

/* Private macro ----------------------------------------------------------------*/

/* Private variables ------------------------------------------------------------*/

static __IO uint32_t uwTick;

/* Private functions ------------------------------------------------------------*/


/**
 *******************************************************************************
 * @brief   Get system tick value.
 * @param   None.
 * @retval  tick value.
 ******************************************************************************/
uint32_t GetTick(void)
{
  return uwTick;
}

/**
 *******************************************************************************
 * @brief   System tick interrupt service function.
 * @param   None.
 * @retval  tick value.
 ******************************************************************************/
void IncTick(void)
{
   uwTick++;
}

/**
 *******************************************************************************
 * @brief   Minimum delay (in milliseconds) based on variable incremented.
 * @param   None.
 * @retval  tick value.
 ******************************************************************************/
void DelayMs(uint32_t Delay)
{
  uint32_t tickStart = GetTick();
  while (GetTick() - tickStart < Delay)
    ;
}

/**
 *******************************************************************************
 * @brief   System Clock initialization.
 * @param   None.
 * @retval  None.
 ******************************************************************************/
void SystemClockInit(void)
{
  RCC_ClkInitTypeStruct RCC_ClkInitStruct;
  
  RCC_Unlock();
  
  RCC_ClockStructInit(&RCC_ClkInitStruct);
  RCC_ClkInitStruct.RCC_ClockType = (uint32_t)(RCC_CLOCK_TYPE_HCLK | RCC_CLOCK_TYPE_PCLK);

  RCC_ClkInitStruct.HCLKDivider = RCC_SYSCLK_DIV1;
  RCC_ClkInitStruct.PCLKDivider = RCC_HCLK_DIV1;
  RCC_ClockConfig(&RCC_ClkInitStruct);

  RCC_Lock();
  
  FLASH_Unlock();
  FLASH_PrefetchBufferCmd(ENABLE);
  FLASH_Lock();

}


/**
 *******************************************************************************
 * @brief   System tick initialization.
 * @param   None.
 * @retval  None.
 ******************************************************************************/
void SystickInit(void)
{
  /* setup systick timer for 1000Hz interrupts */
  if (SysTick_Config(RCC_GetSYSCLKFrequency() / 1000U))
  {
    /* capture error */
    while (1)
    {
    }
  }
  /* configure the systick handler priority */
  NVIC_SetPriority(SysTick_IRQn, 0x00U);
}

/**
  * @}
  */

/**
  * @}
  */

/************************ (C) COPYRIGHT DP *****END OF FILE ***************************/
