/**
 *******************************************************************************
 * @file    dpm32m0xx_dev.h
 *
 * @brief
 *
 * @author  DPM
 *
 * @version V1.0.0
 *
 * @date    2023-11-01
 *
 *******************************************************************************/

#ifndef __DPM32M0XX_DES_H__
#define __DPM32M0XX_DES_H__


#ifdef __cplusplus
extern "C" {
#endif

#if defined(__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050)
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wpadded"
#endif


#include "dpm32m0xx.h"


/**
 ******************************************************************************
 * @brief Device RAM size enum.
 *****************************************************************************/
typedef enum
{
  DES_RAM_SIZE_1KB            = 0x00LU,   /*!< RAM is 1KB.   */
  DES_RAM_SIZE_2KB,                       /*!< RAM is 2KB.   */
  DES_RAM_SIZE_4KB,                       /*!< RAM is 4KB.   */
  DES_RAM_SIZE_8KB,                       /*!< RAM is 8KB.   */
  DES_RAM_SIZE_16KB,                      /*!< RAM is 16KB.  */
  DES_RAM_SIZE_32KB,                      /*!< RAM is 32KB.  */
  DES_RAM_SIZE_64KB,                      /*!< RAM is 64KB.  */
  DES_RAM_SIZE_128KB                      /*!< RAM is 128KB. */
} DES_RAMSizeEnum;

/**
 ******************************************************************************
 * @brief Device prod type enum.
 *****************************************************************************/
typedef enum
{
  DES_PROD_TYPE_F             = 0x00LU,   /*!< F series, general purpose.       */
  DES_PROD_TYPE_L,                        /*!< L series, low power consumption. */
  DES_PROD_TYPE_H,                        /*!< H series, high performance.      */
  DES_PROD_TYPE_M,                        /*!< M series, motor driven.          */
} DES_ProdTypeEnum;

/**
 ******************************************************************************
 * @brief Device pin number enum.
 *****************************************************************************/
typedef enum
{
  DES_PIN_NUM_8               = 0x00LU,   /*!< pins is 8.   */
  DES_PIN_NUM_16,                         /*!< pins is 16.  */
  DES_PIN_NUM_20,                         /*!< pins is 20.  */
  DES_PIN_NUM_24,                         /*!< pins is 24.  */
  DES_PIN_NUM_28,                         /*!< pins is 28.  */
  DES_PIN_NUM_32,                         /*!< pins is 32.  */
  DES_PIN_NUM_40,                         /*!< pins is 40.  */
  DES_PIN_NUM_48,                         /*!< pins is 48.  */
  DES_PIN_NUM_52,                         /*!< pins is 52.  */
  DES_PIN_NUM_64,                         /*!< pins is 64.  */
  DES_PIN_NUM_100,                        /*!< pins is 100. */
  DES_PIN_NUM_144,                        /*!< pins is 144. */
} DES_PinNumEnum;

/**
 ******************************************************************************
 * @brief Device flash size enum.
 *****************************************************************************/
typedef enum
{ 
  DES_FLASH_SIZE_1KB          = 0x00UL,    /*!< FLASH is 1KB.   */
  DES_FLASH_SIZE_2KB,                      /*!< FLASH is 2KB.   */
  DES_FLASH_SIZE_4KB,                      /*!< FLASH is 4KB.   */
  DES_FLASH_SIZE_8KB,                      /*!< FLASH is 8KB.   */
  DES_FLASH_SIZE_16KB,                     /*!< FLASH is 16KB.  */
  DES_FLASH_SIZE_32KB,                     /*!< FLASH is 32KB.  */
  DES_FLASH_SIZE_64KB,                     /*!< FLASH is 64KB.  */
  DES_FLASH_SIZE_128KB                     /*!< FLASH is 128KB. */
} DES_FlashSizeEnum;

/**
 ******************************************************************************
 * @brief Device pack type enum.
 *****************************************************************************/
typedef enum
{
  DES_PACK_TYPE_TSSOP          = 0x00UL,   /*!< Pack type is TSSOP. */
  DES_PACK_TYPE_SOP,                       /*!< Pack type is SOP.   */
  DES_PACK_TYPE_LQPF,                      /*!< Pack type is LQPF.  */
  DES_PACK_TYPE_QPN,                       /*!< Pack type is QPN.   */
} DES_PackTypeEnum;

/**
 ******************************************************************************
 * @brief Integrated Driver Type.
 *****************************************************************************/
typedef enum
{
  DES_GATE_DRIVER_NONE         = 0x00,     /*!< No driver.        */
  DES_GATE_DRIVER_6N           = 0x01,     /*!< 6N driver.        */ 
  DES_GATE_DRIVER_3P3N         = 0x02,     /*!< 3P3N driver.      */
  DES_GATE_DRIVER_3P3N_MOS     = 0x03,     /*!< 3P3N driver + MOS.*/
  DES_GATE_DRIVER_6N_MOS       = 0x04,     /*!< 6N driver + MOS.  */
} DES_GateDriverEnum;

/**
 ******************************************************************************
 * @brief Integrated Power Type.
 *****************************************************************************/
typedef enum
{
  DES_INT_LDO_NONE             = 0x00,      /*!< No LDO.        */
  DES_INT_LDO_5                = 0x01,      /*!< 5V LDO.        */ 
  DES_INT_LDO_3V3              = 0x02,      /*!< 3.3V LDO.      */
  DES_INT_LDO_12_5             = 0x03,      /*!< 12V+5V LDO.    */
  DES_INT_LDO_12_3V3           = 0x04,      /*!< 12V+3.3V LDO.  */
  DES_INT_LDO_15_5             = 0x05,      /*!< 15V+5V LDO.    */ 
  DES_INT_LDO_15_3V3           = 0x07,      /*!< 15V+3.3V LDO.  */
} DES_IntLDOEnum;

/**
 ******************************************************************************
 * @brief Operating Temperature Range.
 *****************************************************************************/
typedef enum
{
  DES_TEMP_N40_P125            = 0x03UL,    /*!< Operating Temperature:-40 to 125. */
  DES_TEMP_N40_P85             = 0x06UL,    /*!< Operating Temperature:-40 to 85.  */
  DES_TEMP_N40_P105            = 0x07UL,    /*!< Operating Temperature:-40 to 105. */
} DES_TemperatureEnum;

/**
 ******************************************************************************
 * @brief Device information struct.
 *****************************************************************************/
typedef struct
{
  uint32_t                  DES_UID[3];           /*!< Unique device identifier.                                */
  DES_ProdTypeEnum          DES_ProdType;         /*!< Product type.                                                       
                                                       This parameter can be a value of @ref DES_ProdTypeEnum.  */
  uint16_t                  DES_SubFamily;        /*!< device sub family coding. */
  DES_PinNumEnum            DES_PinNumer;         /*!< Pin numer.
                                                       This parameter can be a value of @ref DES_PinNumEnum.    */
  DES_FlashSizeEnum         DES_FlashSize;        /*!< Flash size.
                                                       This parameter can be a value of @ref DES_FlashSizeEnum. */
  DES_PackTypeEnum          DES_PackType;         /*!< Pack type.
                                                       This parameter can be a value of @ref DES_PackTypeEnum.  */
  DES_GateDriverEnum        DES_GateDriver;       /*!< Integrated Driver Type.
                                                       This parameter can be a value of @ref DES_GateDriverEnum.*/
  DES_IntLDOEnum            DES_IntLDO;           /*!< Integrated Power Type.
                                                       This parameter can be a value of @ref DES_IntLDOEnum.    */
  uint8_t                   DES_IntVersion;       /*!< Integrated Driver/Power Version Information.             */
  DES_TemperatureEnum       DES_Temperature;      /*!< Integrated Driver Type.
                                                       This parameter can be a value of @ref DES_TemperatureEnum.*/
  uint8_t                   DES_PackVersion;      /*!< Package Pinout Version Information.                      */
  uint8_t                   DES_MCUVersion;       /*!< MCU Version Information.                                 */
  DES_RAMSizeEnum           DES_RAMSize;          /*!< SRAM size.
                                                       This parameter can be a value of @ref DES_RAMSizeEnum.   */
} DES_DevInfoTypeStruct;


/* Get device infomation functions. */
uint32_t                DES_GetUIDWord0(void);
uint32_t                DES_GetUIDWord1(void);
uint32_t                DES_GetUIDWord2(void);

DES_RAMSizeEnum         DES_GetRAMSize(void);
DES_ProdTypeEnum        DES_GetProdType(void);
uint16_t                DES_GetSubFamily(void);
DES_PinNumEnum          DES_GetPinNumer(void);
DES_FlashSizeEnum       DES_GetFlashSize(void);
DES_PackTypeEnum        DES_GetPackType(void);

DES_GateDriverEnum      DES_GetGateDriver(void);
DES_IntLDOEnum          DES_GetIntLDO(void);
DES_TemperatureEnum     DES_GetTemperature(void);
uint8_t                 DES_GetIntVersion(void);
uint8_t                 DES_GetPackVersion(void);
uint8_t                 DES_GetMCUVersion(void);

void                    DES_GetDevInfo(DES_DevInfoTypeStruct* DES_DevInfoType);


#if defined(__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050)
#pragma clang diagnostic pop
#endif

#ifdef __cplusplus
}

#endif

#endif /* __DPM32M0XX_DES_H__ */
