/**
 *******************************************************************************
 * @file    dpm32m0xx_adc.h
 *
 * @brief   Header file for ADC firmware driver.
 *
 * @author  DPM
 *
 * @version V1.0.0
 *
 * @date    2023-11-01
 *
 *******************************************************************************/

#ifndef __DPM32M0XX_ADC_H__
#define __DPM32M0XX_ADC_H__


#ifdef __cplusplus
extern "C" {
#endif

#if defined(__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050)
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wpadded"
#endif


#include "dpm32m0xx.h"

/**
 ******************************************************************************
 * @brief ADC sample time enumeration.
 *****************************************************************************/
#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)   
typedef enum
{
  ADC_SAMPLE_TIME_34Cycles      = 0x00UL,    /*!< ADC channel sample 32+2 ADC work cycles times.*/
  ADC_SAMPLE_TIME_36Cycles      = 0x01UL,    /*!< ADC channel sample 32+4 ADC work cycles times.*/
  ADC_SAMPLE_TIME_40Cycles      = 0x02UL,    /*!< ADC channel sample 32+8 ADC work cycles times.*/
  ADC_SAMPLE_TIME_48Cycles      = 0x03UL,    /*!< ADC channel sample 32+16 ADC work cycles times.*/
  ADC_SAMPLE_TIME_64Cycles      = 0x04UL,    /*!< ADC channel sample 32+32 ADC work cycles times.*/
  ADC_SAMPLE_TIME_96Cycles      = 0x05UL,    /*!< ADC channel sample 32+64 ADC work cycles times.*/
  ADC_SAMPLE_TIME_160Cycles     = 0x06UL,    /*!< ADC channel sample 32+128 ADC work cycles times.*/
  ADC_SAMPLE_TIME_288Cycles     = 0x07UL     /*!< ADC channel sample 32+256 ADC work cycles times.*/
} ADC_SampleTimeEnum;

#define IS_ADC_SAMPLE_TIME(x)     (((x) == ADC_SAMPLE_TIME_34Cycles) || \
                                   ((x) == ADC_SAMPLE_TIME_36Cycles) || \
                                   ((x) == ADC_SAMPLE_TIME_40Cycles) || \
                                   ((x) == ADC_SAMPLE_TIME_48Cycles) || \
                                   ((x) == ADC_SAMPLE_TIME_64Cycles) || \
                                   ((x) == ADC_SAMPLE_TIME_96Cycles) || \
                                   ((x) == ADC_SAMPLE_TIME_160Cycles)|| \
                                   ((x) == ADC_SAMPLE_TIME_288Cycles))

#else /* DPM32M08x || DPM32M05x || DPM32M03x */
typedef enum
{
  ADC_SAMPLE_TIME_2Cycles      = 0x00UL,    /*!< ADC channel sample 2 ADC work cycles times.*/
  ADC_SAMPLE_TIME_4Cycles      = 0x01UL,    /*!< ADC channel sample 4 ADC work cycles times.*/
  ADC_SAMPLE_TIME_8Cycles      = 0x02UL,    /*!< ADC channel sample 8 ADC work cycles times.*/
  ADC_SAMPLE_TIME_16Cycles     = 0x03UL,    /*!< ADC channel sample 16 ADC work cycles times.*/
  ADC_SAMPLE_TIME_32Cycles     = 0x04UL,    /*!< ADC channel sample 32 ADC work cycles times.*/
  ADC_SAMPLE_TIME_64Cycles     = 0x05UL,    /*!< ADC channel sample 64 ADC work cycles times.*/
  ADC_SAMPLE_TIME_128Cycles    = 0x06UL,    /*!< ADC channel sample 128 ADC work cycles times.*/
  ADC_SAMPLE_TIME_256Cycles    = 0x07UL     /*!< ADC channel sample 256 ADC work cycles times.*/
} ADC_SampleTimeEnum;

#define IS_ADC_SAMPLE_TIME(x)     (((x) == ADC_SAMPLE_TIME_2Cycles) || \
                                   ((x) == ADC_SAMPLE_TIME_4Cycles) || \
                                   ((x) == ADC_SAMPLE_TIME_8Cycles) || \
                                   ((x) == ADC_SAMPLE_TIME_16Cycles) || \
                                   ((x) == ADC_SAMPLE_TIME_32Cycles) || \
                                   ((x) == ADC_SAMPLE_TIME_64Cycles) || \
                                   ((x) == ADC_SAMPLE_TIME_128Cycles) || \
                                   ((x) == ADC_SAMPLE_TIME_256Cycles))
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */

/**
 ******************************************************************************
 * @brief ADC hardware trigger select enumeration.
 *****************************************************************************/
typedef enum
{
  ADC_HW_TRIG0 = 0x00UL,    /*!< ADC hardware trigger 0.*/
  ADC_HW_TRIG1              /*!< ADC hardware trigger 1.*/
} ADC_HWTrigX;


/**
 ******************************************************************************
 * @brief ADC data align enumeration.
 *****************************************************************************/
typedef enum
{
  ADC_DATA_ALIGN_RIGHT = 0x00UL,  /*!< ADC data right aligned.*/
  ADC_DATA_ALIGN_LEFT             /*!< ADC data left aligned.*/
} ADC_DataAlignEnum;


/**
 ******************************************************************************
 * @brief ADC data sign enumeration.
 *****************************************************************************/
typedef enum
{
  ADC_DATA_UNSIGNED = 0x00UL,   /*!< ADC data has no signed bits.*/
  ADC_DATA_SIGNED               /*!< ADC data has signed bits.*/
} ADC_DataSignEnum;


/**
 ******************************************************************************
 * @brief ADC channel sample time select enumeration.
 *****************************************************************************/
typedef enum
{
  ADC_SAMPLE_TIME_SEL0 = 0x00UL,  /*!< ADC channel sample time select 0.*/
  ADC_SAMPLE_TIME_SEL1            /*!< ADC channel sample time select 1.*/
} ADC_SampleTimeSelEnum;


/**
 ******************************************************************************
 * @brief ADC common initialization structure definition.
 *****************************************************************************/
typedef struct
{
  FunctionalState SyncTrig;           /*!< Software synchronization triggers.*/

  uint8_t AdcSpeed;                   /*!< ADC comparison current.default 1*/

  uint32_t Vref;                      /*!< ADC reference voltage source select. @ref ADC_Vref*/

  uint32_t ClkDiv;                    /*!< ADC CLK prescaler.This parameters can be a value of @ref ADC_Clk_Div */

  uint32_t PGASel;                    /*!< ADC PGA enable.This parameters can be select one or more from @ref ADC_PGASelEnum */

#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)   
  uint32_t IN15Sel;                   /*!< ADC IN15 source selection.This parameters can be select one or more from @ref ADC_IN15SelEnum */
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */

  ADC_SampleTimeEnum SampleTime0;     /*!< ADC sample time selection 0.
                                            This parameter can be a value of @ref ADC_SampleTimeEnum. */

  ADC_SampleTimeEnum SampleTime1;     /*!< ADC sample time selection 1.
                                            This parameter can be a value of @ref ADC_SampleTimeEnum. */

  uint32_t OversampleRatio;           /*!< ADC oversample ratio. This parameter can be a value of @ref ADC_Oversample_Ratio */

  uint32_t OversampleAccuracy;        /*!< ADC oversampling accuracy. @ref ADC_Oversample_Accuracy */

  uint32_t SampleAccuracy;            /*!< ADC sampling accuracy. @ref ADC_Sample_Accuracy */

} ADCCOM_InitTypeStruct;



/**
 ******************************************************************************
 * @brief ADC commom channel initialization structure definition.
 *****************************************************************************/
typedef struct
{
  uint8_t Channel;                      /*!< ADC channel @ref ADC_Channel */

#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x)
  FunctionalState BufferEnable;         /*!< ADC input buffer enable or disable */
#endif /* DPM32M08x || DPM32M05x || DPM32M03x */

  FunctionalState OversampleEnable;     /*!< ADC over sampling enable or disable */

  ADC_SampleTimeSelEnum SampleTimeSel;  /*!< ADC sampling time select @ref ADC_SampleTimeSelEnum */

  uint16_t Offset;                      /*!< ADC channel offset.This parameter can be a number between 0x0000 and 0xFFF */
} ADCCOM_ChannelStruct;



/**
 ******************************************************************************
 * @brief ADC hardware trigger structure definition.
 *****************************************************************************/
typedef struct
{
  uint32_t TrigSource;          /*!< ADC hardware trigger source. @ref ADC_Trig_Source. */

  uint32_t TrigChannel;         /*!< ADC hardware trigger channel. @ref ADC_Channel. */

  FunctionalState TrigEnable;   /*!< ADC hardware trigger enable. @ref FunctionalState. */

} ADC_HWTrigStruct;


/**
 ******************************************************************************
 * @brief ADC initialization structure definition.
 *****************************************************************************/
typedef struct
{
  uint32_t TrigMode;                 /*!< ADC work mode.This parameters can be a value of @ref ADC_Trig_Mode */

  uint32_t ContinueWait;             /*!< Software trigger channel continuous data sampling waiting time.
                                           This parameters can be a value of @ref ADC_Continue_Wait */

  uint32_t SoftTrigChannel;          /*!< ADC software trigger single channel sampling. @ref ADC_Channel*/

  ADC_HWTrigStruct HardwareTrig0;    /*!< ADC hardware trigger 0 config struct.*/

  ADC_HWTrigStruct HardwareTrig1;    /*!< ADC hardware trigger 1 config struct.*/

  uint32_t ScanChannelLen;           /*!< ADC scan sequence length.*/

} ADC_InitTypeStruct;


/**
 ******************************************************************************
 * @brief ADC channel initialization structure definition.
 *****************************************************************************/
typedef struct
{
  uint8_t Channel;              /*!< ADC config channel. */

  ADC_DataAlignEnum DataAlign;  /*!< ADC channel data alignment. @ref ADC_DataAlignEnum. */

  ADC_DataSignEnum DataSign;    /*!< ADC channel data sign. @ref ADC_DataSignEnum. */

#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x)
  FunctionalState AWDEnable;    /*!< ADC channel analog watchdog enable. @ref FunctionalState. */
#endif /* DPM32M08x || DPM32M05x || DPM32M03x */
} ADC_ChannelStruct;


#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x)
/**
 ******************************************************************************
 * @brief ADC analog watchdog initialization structure definition.
 *****************************************************************************/
typedef struct
{
  uint16_t LowerThreshold;              /*!< ADC analog watchdog lower threshold.
                                              This parameter can be a number between 0x0000 and 0xFFFF. */

  uint16_t HighThreshold;               /*!< ADC analog watchdog higher threshold.
                                              This parameter can be a number between 0x0000 and 0xFFFF. */

  FunctionalState AWDLowerLimitEnable;  /*!< ADC analog watchdog lower limit enable. @ref FunctionalState. */

  FunctionalState AWDUpperLimitEnable;  /*!< ADC analog watchdog upper limit enable. @ref FunctionalState. */


  FunctionalState AWDWindowLimitEnable; /*!< ADC analog watchdog window limit enable. @ref FunctionalState. */
} ADC_AnalogWatchdogStruct;
#endif /* DPM32M08x || DPM32M05x || DPM32M03x */


/**
  * @defgroup ADC_Channel
  */
#define ADC_Channel_0                   0x00UL
#define ADC_Channel_1                   0x01UL
#define ADC_Channel_2                   0x02UL
#define ADC_Channel_3                   0x03UL
#define ADC_Channel_4                   0x04UL
#define ADC_Channel_5                   0x05UL
#define ADC_Channel_6                   0x06UL
#define ADC_Channel_7                   0x07UL
#define ADC_Channel_8                   0x08UL
#define ADC_Channel_9                   0x09UL
#define ADC_Channel_10                  0x0AUL
#define ADC_Channel_11                  0x0BUL
#define ADC_Channel_12                  0x0CUL
#define ADC_Channel_13                  0x0DUL
#define ADC_Channel_14                  0x0EUL
#define ADC_Channel_15                  0x0FUL
/* DPM32M08x | DPM32M05x | DPM32M03x */
#define ADC_Channel_TEMP                ADC_Channel_14
#define ADC_Channel_VDD                 ADC_Channel_15

#define IS_ADC_CHANNEL(x)              (((x) == ADC_Channel_0) || \
                                        ((x) == ADC_Channel_1) || \
                                        ((x) == ADC_Channel_2) || \
                                        ((x) == ADC_Channel_3) || \
                                        ((x) == ADC_Channel_4) || \
                                        ((x) == ADC_Channel_5) || \
                                        ((x) == ADC_Channel_6) || \
                                        ((x) == ADC_Channel_7) || \
                                        ((x) == ADC_Channel_8) || \
                                        ((x) == ADC_Channel_9) || \
                                        ((x) == ADC_Channel_10) || \
                                        ((x) == ADC_Channel_11) || \
                                        ((x) == ADC_Channel_12) || \
                                        ((x) == ADC_Channel_13) || \
                                        ((x) == ADC_Channel_14) || \
                                        ((x) == ADC_Channel_15) || \
                                        ((x) == ADC_Channel_TEMP) || \
                                        ((x) == ADC_Channel_VDD))



/**
  * @defgroup ADC_SCAN_Channel
  */

#define ADC_SCAN_Channel_0               0x00UL
#define ADC_SCAN_Channel_1               0x01UL
#define ADC_SCAN_Channel_2               0x02UL
#define ADC_SCAN_Channel_3               0x03UL
#define ADC_SCAN_Channel_4               0x04UL
#define ADC_SCAN_Channel_5               0x05UL
#define ADC_SCAN_Channel_6               0x06UL
#define ADC_SCAN_Channel_7               0x07UL
#define ADC_SCAN_Channel_8               0x08UL
#define ADC_SCAN_Channel_9               0x09UL
#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x)
#define ADC_SCAN_Channel_10              0x0AUL
#define ADC_SCAN_Channel_11              0x0BUL
#define ADC_SCAN_Channel_12              0x0CUL
#define ADC_SCAN_Channel_13              0x0DUL
#endif /* DPM32M08x || DPM32M05x || DPM32M03x */

#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)   
#define IS_SCAN_ADC_CHANNEL(x)         (((x) == ADC_SCAN_Channel_0) || \
                                        ((x) == ADC_SCAN_Channel_1) || \
                                        ((x) == ADC_SCAN_Channel_2) || \
                                        ((x) == ADC_SCAN_Channel_3) || \
                                        ((x) == ADC_SCAN_Channel_4) || \
                                        ((x) == ADC_SCAN_Channel_5) || \
                                        ((x) == ADC_SCAN_Channel_6) || \
                                        ((x) == ADC_SCAN_Channel_7) || \
                                        ((x) == ADC_SCAN_Channel_8) || \
                                        ((x) == ADC_SCAN_Channel_9))
#else /* DPM32M08x || DPM32M05x || DPM32M03x */
#define IS_SCAN_ADC_CHANNEL(x)         (((x) == ADC_SCAN_Channel_0) || \
                                        ((x) == ADC_SCAN_Channel_1) || \
                                        ((x) == ADC_SCAN_Channel_2) || \
                                        ((x) == ADC_SCAN_Channel_3) || \
                                        ((x) == ADC_SCAN_Channel_4) || \
                                        ((x) == ADC_SCAN_Channel_5) || \
                                        ((x) == ADC_SCAN_Channel_6) || \
                                        ((x) == ADC_SCAN_Channel_7) || \
                                        ((x) == ADC_SCAN_Channel_8) || \
                                        ((x) == ADC_SCAN_Channel_9) || \
                                        ((x) == ADC_SCAN_Channel_10) || \
                                        ((x) == ADC_SCAN_Channel_11) || \
                                        ((x) == ADC_SCAN_Channel_12) || \
                                        ((x) == ADC_SCAN_Channel_13))
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */


/**
  * @defgroup ADC_Clk_Div
  */

#define ADC_CLK_DIV1                    ((uint32_t)0x00000000UL)       /*!< Set the clock source of ADC CLK as PCLK. */
#define ADC_CLK_DIV2                    ((uint32_t)0x00100000UL)       /*!< Set the clock source of ADC CLK as PCLK / 2. */
#define ADC_CLK_DIV3                    ((uint32_t)0x00200000UL)       /*!< Set the clock source of ADC CLK as PCLK / 3. */
#define ADC_CLK_DIV4                    ((uint32_t)0x00300000UL)       /*!< Set the clock source of ADC CLK as PCLK / 4. */
#define ADC_CLK_DIV5                    ((uint32_t)0x00400000UL)       /*!< Set the clock source of ADC CLK as PCLK / 5. */
#define ADC_CLK_DIV6                    ((uint32_t)0x00500000UL)       /*!< Set the clock source of ADC CLK as PCLK / 6. */
#define ADC_CLK_DIV7                    ((uint32_t)0x00600000UL)       /*!< Set the clock source of ADC CLK as PCLK / 7. */
#define ADC_CLK_DIV8                    ((uint32_t)0x00700000UL)       /*!< Set the clock source of ADC CLK as PCLK / 8. */


#define IS_ADC_PRESC(x)                (((x) == ADC_CLK_DIV1) || \
                                        ((x) == ADC_CLK_DIV2) || \
                                        ((x) == ADC_CLK_DIV3) || \
                                        ((x) == ADC_CLK_DIV4) || \
                                        ((x) == ADC_CLK_DIV5) || \
                                        ((x) == ADC_CLK_DIV6) || \
                                        ((x) == ADC_CLK_DIV7) || \
                                        ((x) == ADC_CLK_DIV8))




/**
  * @defgroup ADC_Vref
  */

#define ADC_VREF_AVDD                  ((uint32_t)0x00000000UL)
#define ADC_VREF_BANDGAP               ((uint32_t)0x01000000UL)


#define IS_ADC_VERF(x)               (((x) == ADC_VREF_AVDD) || \
                                      (x) == ADC_VREF_BANDGAP)



/**
  * @defgroup ADC_INT_TYPE
  */

#define ADC_INT_TYPE_EOS               ((uint32_t)0x00000010UL)   /*!< Div zero interrupt. */
#define ADC_INT_TYPE_EOT0              ((uint32_t)0x00000020UL)   /*!< Repeated trigger interrupt. */
#define ADC_INT_TYPE_EOT1              ((uint32_t)0x00000040UL)   /*!< Instruct error interrupt. */
#define ADC_INT_TYPE_EOC               ((uint32_t)0x00000080UL)   /*!< Run completed interrupt. */
#define ADC_INT_TYPE_EOI               ((uint32_t)0x00000100UL)
#define ADC_INT_TYPE_OVR               ((uint32_t)0x00001000UL)
#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x)
#define ADC_INT_TYPE_AWD               ((uint32_t)0x00002000UL)
#endif /* DPM32M08x || DPM32M05x || DPM32M03x */

#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)   
#define IS_ADC_INTTYPE(x)             (((x) == ADC_INT_TYPE_EOS) || \
                                       ((x) == ADC_INT_TYPE_EOT0) || \
                                       ((x) == ADC_INT_TYPE_EOT1) || \
                                       ((x) == ADC_INT_TYPE_EOC) || \
                                       ((x) == ADC_INT_TYPE_EOI) || \
                                       ((x) == ADC_INT_TYPE_OVR))
#else /* DPM32M08x || DPM32M05x || DPM32M03x */
#define IS_ADC_INTTYPE(x)             (((x) == ADC_INT_TYPE_EOS) || \
                                       ((x) == ADC_INT_TYPE_EOT0) || \
                                       ((x) == ADC_INT_TYPE_EOT1) || \
                                       ((x) == ADC_INT_TYPE_EOC) || \
                                       ((x) == ADC_INT_TYPE_EOI) || \
                                       ((x) == ADC_INT_TYPE_OVR) || \
                                       ((x) == ADC_INT_TYPE_AWD))
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */


/**
  * @defgroup ADC_PGASelEnum
  */
#if defined (DPM32M08x) || defined (DPM32M05x)
#define ADC_PGA_SEL_NONE              ((uint32_t)0X00000000UL)
#define ADC_PGA_SEL_0                 ((uint32_t)0X00010000UL)
#define ADC_PGA_SEL_1                 ((uint32_t)0X00020000UL)
#define ADC_PGA_SEL_2                 ((uint32_t)0X00040000UL)
#define ADC_PGA_SEL_3                 ((uint32_t)0X00080000UL)
#define ADC_PGA_SEL_ALL               ((uint32_t)0X000F0000UL)
#elif defined (DPM32M03x)
#define ADC_PGA_SEL_NONE              ((uint32_t)0X00000000UL)
#define ADC_PGA_SEL_0                 ((uint32_t)0X00010000UL)
#define ADC_PGA_SEL_1                 ((uint32_t)0X00020000UL)
#define ADC_PGA_SEL_3                 ((uint32_t)0X00080000UL)
#define ADC_PGA_SEL_ALL               ((uint32_t)0X000B0000UL)
#elif defined (DPM32M036) 
#define ADC_PGA_SEL_NONE              ((uint32_t)0X00000000UL)
#define ADC_PGA_SEL_0                 ((uint32_t)0X00010000UL)
#define ADC_PGA_SEL_1                 ((uint32_t)0X00020000UL)
#define ADC_PGA_SEL_2                 ((uint32_t)0X00040000UL)
#define ADC_PGA_SEL_ALL               ((uint32_t)0X00070000UL)
#elif defined (DPM32M030) 
#define ADC_PGA_SEL_NONE              ((uint32_t)0X00000000UL)
#define ADC_PGA_SEL_0                 ((uint32_t)0X00010000UL)
#define ADC_PGA_SEL_1                 ((uint32_t)0X00020000UL)
#define ADC_PGA_SEL_ALL               ((uint32_t)0X00070000UL)
#elif defined (DPM32M015)   
#define ADC_PGA_SEL_NONE              ((uint32_t)0X00000000UL)
#define ADC_PGA_SEL_0                 ((uint32_t)0X00010000UL)
#define ADC_PGA_SEL_ALL               ((uint32_t)0X00070000UL)
#endif  /* DPM32M08x || DPM32M05x  */

/**
  * @defgroup ADC_IN15SelEnum
  */
#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)   
#define ADC_IN15_SEL_VSS               ((uint32_t)0X00000000UL)
#define ADC_IN15_SEL_BANDGAP           ((uint32_t)0X00000002UL)
#define ADC_IN15_SEL_TEMP              ((uint32_t)0X00000003UL)
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */



#if defined (DPM32M08x) || defined (DPM32M05x)
#define IS_ADC_PGA_SEL(x)             (((x) == ADC_PGA_SEL_NONE) || \
                                       ((x) == ADC_PGA_SEL_0) || \
                                       ((x) == ADC_PGA_SEL_1) || \
                                       ((x) == ADC_PGA_SEL_2) || \
                                       ((x) == ADC_PGA_SEL_3) || \
                                       ((x) == ADC_PGA_SEL_ALL))
#elif defined (DPM32M03x)
#define IS_ADC_PGA_SEL(x)             (((x) == ADC_PGA_SEL_NONE) || \
                                       ((x) == ADC_PGA_SEL_0) || \
                                       ((x) == ADC_PGA_SEL_1) || \
                                       ((x) == ADC_PGA_SEL_3) || \
                                       ((x) == ADC_PGA_SEL_ALL))
#elif defined (DPM32M036)
#define IS_ADC_PGA_SEL(x)             (((x) == ADC_PGA_SEL_NONE) || \
                                       ((x) == ADC_PGA_SEL_0) || \
                                       ((x) == ADC_PGA_SEL_1) || \
                                       ((x) == ADC_PGA_SEL_2) || \
                                       ((x) == ADC_PGA_SEL_ALL))
#elif defined (DPM32M030)
#define IS_ADC_PGA_SEL(x)             (((x) == ADC_PGA_SEL_NONE) || \
                                       ((x) == ADC_PGA_SEL_0) || \
                                       ((x) == ADC_PGA_SEL_1) || \
                                       ((x) == ADC_PGA_SEL_ALL))
#elif defined (DPM32M015)
#define IS_ADC_PGA_SEL(x)             (((x) == ADC_PGA_SEL_NONE) || \
                                       ((x) == ADC_PGA_SEL_0) || \
                                       ((x) == ADC_PGA_SEL_ALL))
#endif  /* DPM32M08x || DPM32M05x  */


#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)   
#define IS_ADC_IN15_SEL(x)             (((x) == ADC_IN15_SEL_VSS) || \
                                        ((x) == ADC_IN15_SEL_BANDGAP) || \
                                        ((x) == ADC_IN15_SEL_TEMP))
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */

/**
  * @defgroup ADC_Oversample_Ratio
  */

#define ADC_Oversample_Ratio_4x          ((uint32_t)0x00000000UL)
#define ADC_Oversample_Ratio_16x         ((uint32_t)0x20000000UL)
#define ADC_Oversample_Ratio_64x         ((uint32_t)0x40000000UL)
#define ADC_Oversample_Ratio_256x        ((uint32_t)0x60000000UL)


#define IS_ADC_OVERSAMPLE_RATIO(x)       (((x) == ADC_Oversample_Ratio_4x) || \
    ((x) == ADC_Oversample_Ratio_16x) || \
    ((x) == ADC_Oversample_Ratio_64x) || \
    ((x) == ADC_Oversample_Ratio_256x))


/**
  * @defgroup ADC_Oversample_Accuracy
  */

#define ADC_Oversample_Accuracy_12Bit     ((uint32_t)0x00000000UL)
#define ADC_Oversample_Accuracy_13Bit     ((uint32_t)0x10000000UL)
#define ADC_Oversample_Accuracy_14Bit     ((uint32_t)0x14000000UL)
#define ADC_Oversample_Accuracy_15Bit     ((uint32_t)0x18000000UL)
#define ADC_Oversample_Accuracy_16Bit     ((uint32_t)0x1C000000UL)

#define IS_ADC_OVERSAMPLE_ACCURACY(x)     (((x) == ADC_Oversample_Accuracy_12Bit) || \
    ((x) == ADC_Oversample_Accuracy_13Bit) || \
    ((x) == ADC_Oversample_Accuracy_14Bit) || \
    ((x) == ADC_Oversample_Accuracy_15Bit) || \
    ((x) == ADC_Oversample_Accuracy_16Bit))

/**
  * @defgroup ADC_Sample_Accuracy
  */

#define ADC_Sample_Accuracy_6Bit         ((uint32_t)0x00000000UL)
#define ADC_Sample_Accuracy_8Bit         ((uint32_t)0x01000000UL)
#define ADC_Sample_Accuracy_10Bit        ((uint32_t)0x02000000UL)
#define ADC_Sample_Accuracy_12Bit        ((uint32_t)0x03000000UL)

#define IS_ADC_SAMPLING_ACCURACY(x)      (((x) == ADC_Sample_Accuracy_6Bit) || \
    ((x) == ADC_Sample_Accuracy_8Bit) || \
    ((x) == ADC_Sample_Accuracy_10Bit) || \
    ((x) == ADC_Sample_Accuracy_12Bit))

#if defined (DPM32M08x)
#define IS_ADC_PERIPH(PERIPH)            ((PERIPH) == ADC0 || (PERIPH) == ADC1)
#else /* DPM32M05x || DPM32M03x || DPM32M036  || DPM32M030 || DPM32M015 */
#define IS_ADC_PERIPH(PERIPH)            ((PERIPH) == ADC0)
#endif /* DPM32M08x */



/**
  * @defgroup ADC_INT_FLAG
  */

#define ADC_INT_FLAG_EO_SUM              ADC_ISR_EO_SUM_Msk
#define ADC_INT_FLAG_OVR_SUM             ADC_ISR_OVR_SUM_Msk
#define ADC_INT_FLAG_EOS                 ADC_ISR_EOS_Msk
#define ADC_INT_FLAG_EOT0                ADC_ISR_EOT0_Msk
#define ADC_INT_FLAG_EOT1                ADC_ISR_EOT1_Msk
#define ADC_INT_FLAG_EOC                 ADC_ISR_EOC_Msk
#define ADC_INT_FLAG_EOI                 ADC_ISR_EOI_Msk
#define ADC_INT_FLAG_OVR_EOS             ADC_ISR_OVR_EOS_Msk
#define ADC_INT_FLAG_OVR_EOT0            ADC_ISR_OVR_EOT0_Msk
#define ADC_INT_FLAG_OVR_EOT1            ADC_ISR_OVR_EOT1_Msk
#define ADC_INT_FLAG_OVR_EOC             ADC_ISR_OVR_EOC_Msk
#define ADC_INT_FLAG_OVR_EOI             ADC_ISR_OVR_EOI_Msk

#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x)
#define ADC_INT_FLAG_AWD_SUM             ADC_ISR_AWD_SUM_Msk
#define ADC_INT_FLAG_AWD_CH0             ADC_ISR_AWD_CH0_Msk
#define ADC_INT_FLAG_AWD_CH1             ADC_ISR_AWD_CH1_Msk
#define ADC_INT_FLAG_AWD_CH2             ADC_ISR_AWD_CH2_Msk
#define ADC_INT_FLAG_AWD_CH3             ADC_ISR_AWD_CH3_Msk
#define ADC_INT_FLAG_AWD_CH4             ADC_ISR_AWD_CH4_Msk
#define ADC_INT_FLAG_AWD_CH5             ADC_ISR_AWD_CH5_Msk
#define ADC_INT_FLAG_AWD_CH6             ADC_ISR_AWD_CH6_Msk
#define ADC_INT_FLAG_AWD_CH7             ADC_ISR_AWD_CH7_Msk
#define ADC_INT_FLAG_AWD_CH8             ADC_ISR_AWD_CH8_Msk
#define ADC_INT_FLAG_AWD_CH9             ADC_ISR_AWD_CH9_Msk
#define ADC_INT_FLAG_AWD_CH10            ADC_ISR_AWD_CH10_Msk
#define ADC_INT_FLAG_AWD_CH11            ADC_ISR_AWD_CH11_Msk
#define ADC_INT_FLAG_AWD_CH12            ADC_ISR_AWD_CH12_Msk
#define ADC_INT_FLAG_AWD_CH13            ADC_ISR_AWD_CH13_Msk
#define ADC_INT_FLAG_AWD_CH14            ADC_ISR_AWD_CH14_Msk
#define ADC_INT_FLAG_AWD_CH15            ADC_ISR_AWD_CH15_Msk
#endif  /* DPM32M08x || DPM32M05x || DPM32M03x */

#if defined (DPM32M08x) || defined (DPM32M05x)
/**
  * @defgroup ADC_DMA_TYPE
  */

#define ADC_DMA_EOC                      ADC_CFG_EOC_DMA_EN_Msk
#define ADC_DMA_EOS                      ADC_CFG_EOS_DMA_EN_Msk
#define ADC_DMA_EOT0                     ADC_CFG_EOT0_DMA_EN_Msk
#define ADC_DMA_EOT1                     ADC_CFG_EOT1_DMA_EN_Msk

#define IS_ADC_DAM_TYPE(x)                    (((x) == ADC_CFG_EOC_DMA_EN_Msk) || \
    ((x) == ADC_CFG_EOS_DMA_EN_Msk) || \
    ((x) == ADC_CFG_EOT0_DMA_EN_Msk) || \
    ((x) == ADC_CFG_EOT1_DMA_EN_Msk))
#endif  /* DPM32M08x || DPM32M05x */

/**
  * @defgroup ADC_Trig_Mode
  */

#define ADC_TRIG_SW_SINGLE_CONV                ((uint32_t)0x00000000UL)   /*!<ADC software triggered single channel conversion. */
#define ADC_TRIG_SW_SINGLE_CONTINUOUS_CONV     ((uint32_t)0x00000002UL)   /*!<ADC software triggered single channel continuous conversion. */
#define ADC_TRIG_SW_SEQ_CONTINUOUS_CONV        ((uint32_t)0x00000003UL)   /*!<ADC software triggered sequence continuous scan. */
#define ADC_TRIG_HW_SINGLE_CONV                ((uint32_t)0x00000004UL)   /*!<ADC hardware triggered single channel conversion.  */
#define ADC_TRIG_HW_SEQ_CONV                   ((uint32_t)0x00000005UL)   /*!<ADC hardware triggered single sequence scan. */
#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)   
#define ADC_TRIG_SW_SINGLE_SEQ_CONV            ((uint32_t)0x00000001UL)   /*!<ADC software triggered single sequence scan. */
#define ADC_TRIG_HW_SINGLE_SEQ_CONV            ((uint32_t)0x00000006UL)   /*!<ADC hardware triggered single channel and single sequence scan. */
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */

#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)   
#define IS_ADC_TRIG_MODE(x)                   (((x) == ADC_TRIG_SW_SINGLE_CONV) || \
                                               ((x) == ADC_TRIG_SW_SINGLE_CONTINUOUS_CONV) || \
                                               ((x) == ADC_TRIG_SW_SEQ_CONTINUOUS_CONV) || \
                                               ((x) == ADC_TRIG_HW_SINGLE_CONV) || \
                                               ((x) == ADC_TRIG_HW_SEQ_CONV) || \
                                               ((x) == ADC_TRIG_SW_SINGLE_SEQ_CONV) || \
                                               ((x) == ADC_TRIG_HW_SINGLE_SEQ_CONV))
#else /* DPM32M08x || DPM32M05x || DPM32M03x */
#define IS_ADC_TRIG_MODE(x)                   (((x) == ADC_TRIG_SW_SINGLE_CONV) || \
                                               ((x) == ADC_TRIG_SW_SINGLE_CONTINUOUS_CONV) || \
                                               ((x) == ADC_TRIG_SW_SEQ_CONTINUOUS_CONV) || \
                                               ((x) == ADC_TRIG_HW_SINGLE_CONV) || \
                                               ((x) == ADC_TRIG_HW_SEQ_CONV))
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */

/**
  * @defgroup ADC_Continue_Wait
  */
#define ADC_CONTINUE_WAIT_CYCLE_0              ((uint32_t)0x00000000UL)
#define ADC_CONTINUE_WAIT_CYCLE_4              ((uint32_t)0x00000100UL)
#define ADC_CONTINUE_WAIT_CYCLE_8              ((uint32_t)0x00000200UL)
#define ADC_CONTINUE_WAIT_CYCLE_16             ((uint32_t)0x00000300UL)
#define ADC_CONTINUE_WAIT_CYCLE_32             ((uint32_t)0x00000400UL)
#define ADC_CONTINUE_WAIT_CYCLE_64             ((uint32_t)0x00000500UL)
#define ADC_CONTINUE_WAIT_CYCLE_128            ((uint32_t)0x00000600UL)
#define ADC_CONTINUE_WAIT_CYCLE_256            ((uint32_t)0x00000700UL)


#define IS_ADC_CONTINUE_WAIT(x)               (((x) == ADC_CONTINUE_WAIT_CYCLE_0) || \
    ((x) == ADC_CONTINUE_WAIT_CYCLE_4) || \
    ((x) == ADC_CONTINUE_WAIT_CYCLE_8) || \
    ((x) == ADC_CONTINUE_WAIT_CYCLE_16) || \
    ((x) == ADC_CONTINUE_WAIT_CYCLE_32) || \
    ((x) == ADC_CONTINUE_WAIT_CYCLE_64) || \
    ((x) == ADC_CONTINUE_WAIT_CYCLE_128) || \
    ((x) == ADC_CONTINUE_WAIT_CYCLE_256))

/**
  * @defgroup ADC_Trig_Source
  */
#define ADC_TRIG_SOURCE_TIMER0                 0X00UL
#define ADC_TRIG_SOURCE_TIMER1                 0X01UL
#define ADC_TRIG_SOURCE_TIMER2                 0X02UL
#define ADC_TRIG_SOURCE_TIMER3                 0X03UL
#define ADC_TRIG_SOURCE_TIMER4                 0X04UL
#define ADC_TRIG_SOURCE_TIMER5                 0X05UL
#define ADC_TRIG_SOURCE_EXIT_RISING            0X06UL
#define ADC_TRIG_SOURCE_EXIT_FALLING           0X07UL
#define ADC_TRIG_SOURCE_EPWM_TRIG1             0X08UL
#define ADC_TRIG_SOURCE_EPWM_TRIG2             0X09UL

#define IS_ADC_TRIG_SOURCE(x)                 (((x) == ADC_TRIG_SOURCE_TIMER0) || \
    ((x) == ADC_TRIG_SOURCE_TIMER1) || \
    ((x) == ADC_TRIG_SOURCE_TIMER2) || \
    ((x) == ADC_TRIG_SOURCE_TIMER3) || \
    ((x) == ADC_TRIG_SOURCE_TIMER4) || \
    ((x) == ADC_TRIG_SOURCE_TIMER5) || \
    ((x) == ADC_TRIG_SOURCE_EXIT_RISING) || \
    ((x) == ADC_TRIG_SOURCE_EXIT_FALLING) || \
    ((x) == ADC_TRIG_SOURCE_EPWM_TRIG1) || \
    ((x) == ADC_TRIG_SOURCE_EPWM_TRIG2))






/* ADC0 and ADC1 commom configuration */
void                    ADCCOM_DeInit(void);
void                    ADCCOM_Init(ADCCOM_InitTypeStruct *ADCCOM_InitType);
void                    ADCCOM_StructInit(ADCCOM_InitTypeStruct *ADCCOM_InitType);
void                    ADCCOM_ChannelConfig(ADCCOM_ChannelStruct* ADCCOM_Channel);
void                    ADCCOM_ChannelStructInit(ADCCOM_ChannelStruct *ADCCOM_Channel);

/* ADCx Initialization and enable */
void                    ADC_DeInit(ADC_Type* ADCx);
void                    ADC_Init(ADC_Type* ADCx, ADC_InitTypeStruct* ADC_InitType);
void                    ADC_StructInit(ADC_InitTypeStruct *ADC_InitType);
void                    ADC_Cmd(ADC_Type* ADCx, FunctionalState NewState);
void                    ADC_ChannelConfig(ADC_Type* ADCx, ADC_ChannelStruct* ADC_Channel);
void                    ADC_ChannelStructInit(ADC_ChannelStruct *ADC_Channel);
void                    ADC_TrigModeConfig(ADC_Type* ADCx, uint32_t ADC_TrigMode);

/* Interrupts and flags management functions */
void                    ADC_IntCmd(ADC_Type* ADCx, uint32_t ADC_Int, FunctionalState NewState);
FunctionalState         ADC_GetIntCmdStatus(ADC_Type *ADCx, uint32_t ADC_Int);
FlagState               ADC_GetIntFlagStatus(ADC_Type* ADCx, uint32_t ADC_IntFlag);
void                    ADC_ClearIntFlag(ADC_Type* ADCx, uint32_t ADC_IntFlag);

#if defined (DPM32M08x) || defined (DPM32M05x)
/* DMA config functions */
void                    ADC_DMACmd(ADC_Type* ADCx, uint32_t ADC_DMA, FunctionalState NewState);
#endif  /* DPM32M08x || DPM32M05x */

/* Software trigger and single convert functions */
void                    ADC_StartConversion(ADC_Type* ADCx);
void                    ADC_StopConversion(ADC_Type* ADCx);
void                    ADC_SoftwareTrigConfig(ADC_Type* ADCx, uint8_t ADC_Ch);
uint16_t                ADC_GetSingleData(ADC_Type* ADCx);

/* Hardware trigger and config functions */
void                    ADC_HardwareTrigCmd(ADC_Type* ADCx, ADC_HWTrigX HWTrigX, FunctionalState NewState);
void                    ADC_HardwareTrigConfig(ADC_Type* ADCx, ADC_HWTrigX HWTrigX, ADC_HWTrigStruct ADC_HWTrig);
uint16_t                ADC_GetHWTrig0Data(ADC_Type* ADCx);
uint16_t                ADC_GetHWTrig1Data(ADC_Type* ADCx);


/* Inject mode functions */
void                    ADC_InjectChannelConfig(ADC_Type* ADCx, uint8_t INJ_Ch);
void                    ADC_InjectStart(ADC_Type* ADCx);
uint16_t                ADC_GetInjectData(ADC_Type* ADCx);


/* Scan mode functions. */
void                    ADC_ScanChannelConfig(ADC_Type* ADCx, uint8_t ScanChannel, uint8_t ADC_Channel);
uint16_t                ADC_GetScanData(ADC_Type* ADCx, uint8_t Channel);


/* Analog watchdog config functions */
#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x)
void                    ADC_AnalogWatchdogConfig(ADC_Type* ADCx, ADC_AnalogWatchdogStruct* ADC_AWDGConfig);
void                    ADC_AnalogWatchdogChannelCmd(ADC_Type* ADCx, uint8_t channel, FunctionalState NewState);
#endif /* DPM32M08x || DPM32M05x || DPM32M03x */



#if defined(__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050)
#pragma clang diagnostic pop
#endif

#ifdef __cplusplus
}

#endif

#endif /* __DPM32M0XX_ADC_H__ */
