/**
 *******************************************************************************
 * @file    cct_capturefrequency.c
 *
 * @brief   Implement system specific functions.
 *
 * @author  DP
 *
 * @version V1.0.1
 *
 * @date    2024-11-11
 *
 *******************************************************************************/

/*
    Copyright (c) 2024, DP Microelectronics Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

/* Files include */
#include "cct_capturefrequency.h"
#include "systick.h"
#include "dpm32m0xx.h"
#include "stdio.h"
#include "uart_printf.h"

/**
  * @addtogroup DPM32M036_LibSamples
  * @{
  */

/**
  * @addtogroup CCT
  * @{
  */

/**
  * @addtogroup CCT InputCapture
  * @{
  */

/* Private typedef --------------------------------------------------------------*/

/* Private define ---------------------------------------------------------------*/

/* Private macro ----------------------------------------------------------------*/

/* Private variables ------------------------------------------------------------*/
float CH0Time = 0;
uint32_t ReloadValue = 48000;

/* Private functions ------------------------------------------------------------*/
volatile uint32_t TotalCapture = 0;
volatile uint32_t HighLevelCapture = 0;
volatile uint16_t OverFlowCnt = 0;
volatile uint16_t TotalCnt = 0;
volatile uint16_t HighCnt = 0;
void CCT0_IRQ_Handler(void)
{
  if(CCT0->SR & ((uint32_t)CCT_INT_FLAG_OVERFLOW))
  {
    OverFlowCnt++;

    if(OverFlowCnt > 1000)
    {
      TotalCapture = 0;
      HighLevelCapture = 0;
      HighCnt = 0;
      TotalCnt = 0;
      OverFlowCnt = 0;
    }

    CCT0->SR = CCT_INT_FLAG_OVERFLOW;
  }

  if(CCT0->SR & ((uint32_t)CCT_INT_FLAG_CH0_CAPTURE))
  {
    if(CCT0->SR & (CCT_FLAG_CH0_RISING_CAPTURE))
    {

      TotalCapture = CCT0->CH0_VALUE + HighLevelCapture;

      TotalCnt = OverFlowCnt;
      OverFlowCnt = 0;
      CCT0->SR = CCT_FLAG_CH0_RISING_CAPTURE;
    }
    else if(CCT0->SR & (CCT_FLAG_CH0_FALLING_CAPTURE))
    {
      HighLevelCapture = CCT0->CH0_VALUE;
      HighCnt = OverFlowCnt;
      CCT0->SR = CCT_FLAG_CH0_FALLING_CAPTURE;
    }

    CCT0->SR = CCT_INT_FLAG_CH0_CAPTURE;
  }
}

/**
 *******************************************************************************
 * @brief   CCT Config.
 * @param   None.
 * @retval  None.
 ******************************************************************************/
void CCT_Config(void)
{
  /* Enable clock */
  RCC_Unlock();
  RCC_APBPeriphClockCmd(RCC_APB_PERIPH_CCT0, ENABLE);
  RCC_Lock();
  
  /* Config CCT0 */
  CCT_TimeBaseInitTypeStruct CCT_TimeBaseInitStruct;
  CCT_TimeBaseStructInit(&CCT_TimeBaseInitStruct);
  CCT_TimeBaseInitStruct.CCT_CounterCycle = CCT_COUNTER_CYCLE_PERIOD;
  CCT_TimeBaseInitStruct.CCT_ClockDivision = CCT_CLK_DIV1;
  CCT_TimeBaseInitStruct.CCT_ReloadValue = ReloadValue - 1;
  CCT_TimeBaseInit(CCT0, &CCT_TimeBaseInitStruct);
  CCT_Cmd(CCT0, ENABLE);
  /* Config CCT0 CH0  input capture */
  CCT_ICInitTypeStruct CCT_ICInitStruct;
  CCT_ICStructInit(&CCT_ICInitStruct);
  CCT_ICInitStruct.CCT_ICSingalSel = CCT_IC_SINGAL_SOURCE_PIN;
  CCT_ICInitStruct.CCT_ICFiltLen = CCT_IC_FILT_LEN8;
  CCT_ICInitStruct.CCT_ICFiltSampDiv = CCT_IC_FILT_SAMPLE_DIV1;
  CCT_ICInitStruct.CCT_ICAction = CCT_IC_ACTION_CLEAR_CNT;
  CCT_ICInitStruct.CCT_ICPolarity = CCT_IC_POLARITY_RISINGFALLING;
  CCT_IC0Init(CCT0, &CCT_ICInitStruct);
  /* Enable CCT0 CH0 */
  CCT_CH0Cmd(CCT0, ENABLE);

  /* Clear rising interrupt flag*/
  CCT_ClearFlag(CCT0, CCT_FLAG_CH0_RISING_CAPTURE);
  CCT_ClearFlag(CCT0, CCT_FLAG_CH0_FALLING_CAPTURE);
  
  /* CCT capture interrupt */
  NVIC_SetPriority(CCT0_IRQn, 2);
  NVIC_EnableIRQ(CCT0_IRQn);
  
  CCT_IntCmd(CCT0, CCT_INT_TYPE_OVERFLOW, ENABLE);
  CCT_IntCmd(CCT0, CCT_INT_TYPE_CH0_CAP, ENABLE);
}

/**
 *******************************************************************************
 * @brief   CCT GPIO Init.
 * @param   None.
 * @retval  None.
 ******************************************************************************/
void CCT_GPIOInit(void)
{
  /* Enable clock */
  RCC_Unlock();
  RCC_AHBPeriphClockCmd(RCC_AHB_PERIPH_GPIO, ENABLE);
  RCC_Lock();
  
  /* Config CCT0 GPIO */
  GPIO_InitTypeStruct CCT_GPIO_InitStruct;
  GPIO_StructInit(&CCT_GPIO_InitStruct);
  CCT_GPIO_InitStruct.GPIO_Mode = GPIO_MODE_ALT_FUNC;
  CCT_GPIO_InitStruct.GPIO_Speed = GPIO_SPEED_HIGH;
  CCT_GPIO_InitStruct.GPIO_OType = GPIO_OTYPE_PP;
  CCT_GPIO_InitStruct.GPIO_PuPd = GPIO_PUPD_NOPULL;
  CCT_GPIO_InitStruct.GPIO_Pin = GPIO_PIN_2 ;
  GPIO_Init(GPIOA, &CCT_GPIO_InitStruct);
  GPIO_AltFuncConfig(GPIOA, GPIO_PIN_2, GPIO_ALT_FUNC_0);
}

/**
 *******************************************************************************
 * @brief   CCT Get Frequency DutyCycle.
 * @param   None.
 * @retval  None.
 ******************************************************************************/
void CCT_GetFrequencyDutyCycle(void)
{
  uint32_t TotleLevelTime = 0;
  uint32_t HighLevelTime = 0;
  float Frequency = 0;
  float DutyCycle = 0;

  DelayMs(1000);
  TotleLevelTime = TotalCapture + TotalCnt*ReloadValue;
  HighLevelTime = HighLevelCapture + HighCnt*ReloadValue;

  Frequency = 1000 * ((float)ReloadValue / ((TotleLevelTime)));
  DutyCycle = (float)(HighLevelTime) / (TotleLevelTime) * 100;
  printf("Capture freq:%.2f Hz, duty:%.2f%%\r\n", Frequency, DutyCycle);
}
/**
 *******************************************************************************
 * @brief   CCT InputCapture Sample Init.
 * @param   None.
 * @retval  None.
 ******************************************************************************/
void CCT_CaptureFrequency_SampleInit(void)
{
  /* CCT GPIO Init */
  CCT_GPIOInit();
  /* CCT Config */
  CCT_Config();
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/************************ (C) COPYRIGHT DP *****END OF FILE ***************************/
