/**
 *******************************************************************************
 * @file    dpm32m0xx_dbgmcu.c
 *
 * @brief   Source file for DBG firmware driver.
 *          This file provides firmware functions to manage the following :
 *          functionalities of the Debug (DBG) peripheral:
 *           + Peripheral configuration
 *
 * @author  DPM
 *
 * @version V1.0.0
 *
 * @date    2023-11-01
 *
 * @verbatim
 ===============================================================================
                       ##### How to use this driver #####
 ===============================================================================
    [..]
      (#) Unlock register write protect using RCC_Unlock().

      (#) Configuration peripheral functions when debug mode using
          DBG_PeriphCmd() or DBG_PeriphConfig().

 * @endverbatim
 *******************************************************************************/

#include "dpm32m0xx_dbg.h"


/**
 *******************************************************************************
 * @brief   Deinitializes the MCU in debug mode peripheral registers to their
 *          default reset values.
 * @retval  None.
 ******************************************************************************/
void DBG_DeInit(void)
{
  /* Reset debug configuration register. */
#if defined (DPM32M036) || defined (DPM32M030) || defined (DPM32M015)   
  DBG->CFG = (uint32_t)0x0000060F;
#else /* DPM32M08x || DPM32M05x || DPM32M03x */
  DBG->CFG = (uint32_t)0x0000067F;
#endif /* DPM32M036 || DPM32M030 || DPM32M015 */
}

/**
 *******************************************************************************
 * @brief   Enable or disable specifies peripheral function in debug mode.
 * @param   [in]  DBG_Periph: Specifies the peripheral.
 *                This parameter can be a value of @ref DBG_PeriphEnum.
 *                  @arg DBG_PERIPH_WDG: WDG peripheral.
 *                  @arg DBG_PERIPH_TIM0: TIM0 peripheral.
 *                  @arg DBG_PERIPH_TIM1: TIM1 peripheral.
 *                  @arg DBG_PERIPH_TIM2: TIM2 peripheral.
 *                  @arg DBG_PERIPH_TIM3: TIM3 peripheral.
 *                  @arg DBG_PERIPH_TIM4: TIM4 peripheral.
 *                  @arg DBG_PERIPH_TIM5: TIM5 peripheral.
 *                  @arg DBG_PERIPH_CCT0: CCT0 peripheral.
 *                  @arg DBG_PERIPH_CCT1: CCT1 peripheral.
 *                  @arg DBG_PERIPH_LPTIM: LPTIM peripheral.
 *                  @arg DBG_PERIPH_EPWM: EPWM peripheral.
 * @param   [in]  NewState: New state of periph in debug mode.
 *                This parameter can be: ENABLE or DISABLE.
 * @retval  None.
 ******************************************************************************/
void DBG_PeriphCmd(DBG_PeriphEnum DBG_Periph, FunctionalState NewState)
{
  if(DISABLE != NewState)
  {
    /* Enable periph function in debug mode. */
    DBG->CFG |= (uint32_t)DBG_Periph;
  }
  else
  {
    /* Disable periph function in debug mode. */
    DBG->CFG &= ~(uint32_t)DBG_Periph;
  }
}

/**
 *******************************************************************************
 * @brief   Config specifies peripheral counter mode in debug mode.
 * @param   [in]  DBG_Periph: Specifies the peripheral.
 *                This parameter can be a value of @ref DBG_PeriphEnum.
 *                  @arg DBG_PERIPH_WDG: WDG peripheral.
 *                  @arg DBG_PERIPH_TIM0: TIM0 peripheral.
 *                  @arg DBG_PERIPH_TIM1: TIM1 peripheral.
 *                  @arg DBG_PERIPH_TIM2: TIM2 peripheral.
 *                  @arg DBG_PERIPH_TIM3: TIM3 peripheral.
 *                  @arg DBG_PERIPH_TIM4: TIM4 peripheral.
 *                  @arg DBG_PERIPH_TIM5: TIM5 peripheral.
 *                  @arg DBG_PERIPH_CCT0: CCT0 peripheral.
 *                  @arg DBG_PERIPH_CCT1: CCT1 peripheral.
 *                  @arg DBG_PERIPH_LPTIM: LPTIM peripheral.
 *                  @arg DBG_PERIPH_EPWM: EPWM peripheral.
 * @param   [in]  NewState: New state of periph in debug mode.
 *                This parameter can be a value of @ref DEB_PeriphStateEnum.
 *                  @arg DBG_STATE_CONTINUE: periph continue counter.
 *                  @arg DBG_STATE_PAUSE: periph pause counter.
 * @retval  None.
 ******************************************************************************/
void DBG_PeriphConfig(DBG_PeriphEnum DBG_Periph, DBG_PeriphStateEnum NewState)
{
  if(DBG_STATE_CONTINUE != NewState)
  {
    /* Periph pause counter function in debug mode. */
    DBG->CFG |= (uint32_t)DBG_Periph;
  }
  else
  {
    /* Periph continue counter function in debug mode. */
    DBG->CFG &= ~(uint32_t)DBG_Periph;
  }
}
