/**
 *******************************************************************************
 * @file    i2c_masterblock.c
 *
 * @brief   Implement system specific functions.
 *
 * @author  DP
 *
 * @version V1.0.0
 *
 * @date    2024-12-01
 *
 *******************************************************************************/

/*
    Copyright (c) 2024, DP Microelectronics Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

/* Files include */
#include "i2c_masterblock.h"

/**
  * @addtogroup DPM32M015_LibSamples
  * @{
  */

/**
  * @addtogroup I2C
  * @{
  */

/**
  * @addtogroup I2C MasterBlock
  * @{
  */

/* Private typedef --------------------------------------------------------------*/
I2C_InitTypeStruct I2C_InitStruct;
/* Private define ---------------------------------------------------------------*/
#define NUM 5
/* Private macro ----------------------------------------------------------------*/

/* Private variables ------------------------------------------------------------*/
uint8_t tx_buf[NUM] = {0x11, 0x22, 0x44, 0x88, 0xAA};
uint8_t rx_buf[NUM] ;
/* Private functions ------------------------------------------------------------*/

/**
 *******************************************************************************
 * @brief   I2C GPIO Init.
 * @param   None.
 * @retval  None.
 ******************************************************************************/
void I2C_GPIOInit(void)
{
  RCC_Unlock();
  RCC_AHBPeriphClockCmd(RCC_AHB_PERIPH_GPIO, ENABLE);
  RCC_Lock();
  /* Cofig I2C GPIO */
  GPIO_InitTypeStruct GPIO_InitStruct;
  GPIO_StructInit(&GPIO_InitStruct);
  GPIO_InitStruct.GPIO_Mode = GPIO_MODE_ALT_FUNC;
  GPIO_InitStruct.GPIO_Speed = GPIO_SPEED_HIGH;
  GPIO_InitStruct.GPIO_OType = GPIO_OTYPE_OD;
  GPIO_InitStruct.GPIO_PuPd = GPIO_PUPD_NOPULL;
  GPIO_InitStruct.GPIO_Pin = GPIO_PIN_11 | GPIO_PIN_12;
  GPIO_Init(GPIOA, &GPIO_InitStruct);
  GPIO_AltFuncConfig(GPIOA,GPIO_PIN_11, GPIO_ALT_FUNC_5);
  GPIO_AltFuncConfig(GPIOA,GPIO_PIN_12, GPIO_ALT_FUNC_5);
}
/**
 *******************************************************************************
 * @brief   I2C Config.
 * @param   None.
 * @retval  None.
 ******************************************************************************/
void I2C_Config(void)
{
  RCC_Unlock();
  RCC_APBPeriphClockCmd(RCC_APB_PERIPH_I2C, ENABLE);
  RCC_Lock();

  /* Config I2C */
  I2C_StructInit(&I2C_InitStruct);
  I2C_InitStruct.I2C_OwnAddress = 0x00;
  I2C_InitStruct.I2C_Prescaler = 0x16;
  I2C_InitStruct.I2C_Timing = 0x00420F13;
  I2C_InitStruct.I2C_DigitalFilter = 0x0F;
  I2C_InitStruct.I2C_Mode = I2C_MODE_MASTER;
  I2C_InitStruct.I2C_AddressingMode = I2C_ADDRESSING_MODE_7BIT;
  I2C_Init(&I2C_InitStruct);
  /* Enable I2C */
  I2C_Cmd(ENABLE);
}

/**
 *******************************************************************************
 * @brief   I2C Master Transmit.
 * @param   None.
 * @retval  None.
 ******************************************************************************/
void I2C_Master_Transmit(uint16_t address, uint8_t *buf, uint16_t len)
{
  uint32_t timeout=1000;
  uint32_t time = GetTick() + timeout;
  uint32_t cnt = 0;
  /* Clear send buffer */
  I2C_ClearTxBuffer();
  /* Config target address */
  I2C_TargetAddressConfig(address);
  /* Config direction */
  I2C_TransferDirectionConfig(I2C_DIRECTION_TRANSMITTER);
  /* Send START signal */
  I2C_GenerateSTART();

  /* Waiting for the START signal end flag */
  while(I2C_GetIntFlagStatus(I2C_INT_FLAG_STARTF) != SET && (time > GetTick()));

  if(time <= GetTick())
  {
    goto TRANSMIT_ERR;
  }

  /* Clear START signal end flag */
  I2C_ClearIntFlag(I2C_INT_FLAG_STARTF);

  /* Send data */
  while(cnt < len)
  {
    I2C_SendData(buf[cnt++]);

    while(I2C_GetIntFlagStatus(I2C_INT_FLAG_TXE) != SET && (time > GetTick()));

    if(time <= GetTick())
    {
      goto TRANSMIT_ERR;
    }
  }

  /* Waiting for sending completion flag */
  while(I2C_GetIntFlagStatus(I2C_INT_FLAG_TXE) != SET && (time > GetTick()));

  if(time <= GetTick())
  {
    goto TRANSMIT_ERR;
  }

  I2C_ClearIntFlag(I2C_INT_FLAG_TXE);
  /* Send STOP signal */
  I2C_GenerateSTOP();

  /* Waiting for the STOP signal end flag */
  while(I2C_GetIntFlagStatus(I2C_INT_FLAG_STOPF) != SET && (time > GetTick()));

  if(time <= GetTick())
  {
    goto TRANSMIT_ERR;
  }

  /* Clear STOP flag */
  I2C_ClearIntFlag(I2C_INT_FLAG_STOPF);

TRANSMIT_ERR:
  I2C_GenerateSTOP();
}
/**
 *******************************************************************************
 * @brief   I2C Master Receive.
 * @param   None.
 * @retval  None.
 ******************************************************************************/
void I2C_Master_Receive(uint16_t address, uint8_t *buf, uint16_t len)
{
  uint32_t timeout=1000;
  uint32_t time = GetTick() + timeout;
  uint32_t cnt = 0;
  /* Config target address */
  I2C_TargetAddressConfig(address);
  /* Config direction */
  I2C_TransferDirectionConfig(I2C_DIRECTION_RECEIVER);
  /* Config bytes received */
  I2C_NumberOfBytesConfig(len);
  /* Send START signal */
  I2C_GenerateSTART();

  /* Waiting for the START signal end flag */
  while(I2C_GetIntFlagStatus(I2C_INT_FLAG_STARTF) != SET && (time > GetTick()));

  if(time <= GetTick())
  {
    goto RECEIVE_ERR;
  }

  /* Clear START signal end flag */
  I2C_ClearIntFlag(I2C_INT_FLAG_STARTF);

  /* Receive data */
  while(cnt < len)
  {
    /* Receive data buffer is not empty detection flag */
    if(I2C_GetIntFlagStatus(I2C_INT_FLAG_RXNE) == SET)
    {
      buf[cnt++] = I2C_ReceiveData();
    }

    if(time <= GetTick())
    {
      goto RECEIVE_ERR;
    }
  }

  /* Send STOP signal */
  I2C_GenerateSTOP();

  /* Waiting for the STOP signal end flag */
  while(I2C_GetIntFlagStatus(I2C_INT_FLAG_STOPF) != SET && (time > GetTick()));

  if(time <= GetTick())
  {
    goto RECEIVE_ERR;
  }

  /* Clear STOP flag */
  I2C_ClearIntFlag(I2C_INT_FLAG_STOPF);

RECEIVE_ERR:
  I2C_GenerateSTOP();
}
/**
 *******************************************************************************
 * @brief   I2C Master Block Sample.
 * @param   None.
 * @retval  None.
 ******************************************************************************/
void I2C_MasterBlock_Sample(void)
{
  /* GPIO init */
  I2C_GPIOInit();

  /* I2C config */
  I2C_Config();  
  
  /* I2C transmit */
  I2C_Master_Transmit(0x00, tx_buf, 5);

  /* I2C receive */
  I2C_Master_Receive(0x00, rx_buf, 5);
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/************************ (C) COPYRIGHT DP *****END OF FILE ***************************/
