/**
 *******************************************************************************
 * @file    spi_masterinterrupt.c
 *
 * @brief   Implement system specific functions.
 *
 * @author  DP
 *
 * @version V1.0.0
 *
 * @date    2024-12-01
 *
 *******************************************************************************/

/*
    Copyright (c) 2024, DP Microelectronics Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

/* Files include */
#include "spi_slaveinterrupt.h"

/**
  * @addtogroup DPM32M015_LibSamples
  * @{
  */

/**
  * @addtogroup SPI
  * @{
  */

/**
  * @addtogroup SPI SlaveInterrupt
  * @{
  */

/* Private typedef --------------------------------------------------------------*/

/* Private define ---------------------------------------------------------------*/
uint16_t spi_tx_buf[5] = {0x11, 0x22, 0x44, 0x88, 0xAA};
uint16_t spi_rx_buf[5] ;
uint8_t rxXferCnt = 5;
uint8_t txXferCnt = 5;
uint8_t int_flag_cnt_rxne = 0;
uint8_t int_flag_cnt_txe = 0;

/* Private macro ----------------------------------------------------------------*/

/* Private variables ------------------------------------------------------------*/

/* Private functions ------------------------------------------------------------*/
/**
 *******************************************************************************
 * @brief   SPI IRQHandler.
 * @param   None.
 * @retval  None.
 ******************************************************************************/
void SPI0_IRQ_Handler(void)
{
  if(SPI_GetIntCmdStatus(SPI0, SPI_INT_TYPE_TXE) != DISABLE && SPI_GetIntFlagStatus(SPI0, SPI_INT_FLAG_TXE) != RESET)
  {
    SPI_ClearIntFlag(SPI0, SPI_INT_FLAG_TXE);

    if(int_flag_cnt_txe < 5)
    {
      SPI_SendData(SPI0, spi_tx_buf[int_flag_cnt_txe]);
    }

    int_flag_cnt_txe++;
  }

  if(SPI_GetIntCmdStatus(SPI0, SPI_INT_TYPE_RXNE) != DISABLE && SPI_GetIntFlagStatus(SPI0, SPI_INT_FLAG_RXNE) != RESET)
  {
    SPI_ClearIntFlag(SPI0, SPI_INT_FLAG_RXNE);

    if(int_flag_cnt_rxne < 5)
    {
      spi_rx_buf[int_flag_cnt_rxne] = SPI_ReceiveData(SPI0);
    }

    int_flag_cnt_rxne++;
  }
}

/**
 *******************************************************************************
 * @brief   SPI config Init.
 * @param   None.
 * @retval  None.
 ******************************************************************************/
void SPI_Config(void)
{
  /* Enable clock */
  RCC_Unlock();
  RCC_APBPeriphClockCmd(RCC_APB_PERIPH_SPI0, ENABLE);
  RCC_Lock();
  
  SPI_InitTypeStruct SPI_InitStruct;
  SPI_StructInit(&SPI_InitStruct);
  /* Config SPI0 */
  SPI_InitStruct.SPI_Mode = SPI_MODE_SLAVE;
  SPI_InitStruct.SPI_DataSize = SPI_DATASIZE_8BIT;
  SPI_InitStruct.SPI_CPOL = SPI_CPOL_LOW;
  SPI_InitStruct.SPI_CPHA = SPI_CPHA_1EDGE;
  SPI_InitStruct.SPI_BaudRatePrescaler = SPI_BAUDRATE_PRESCALER_128;
  SPI_InitStruct.SPI_FirstBit = SPI_FRIST_BIT_MSB;
  SPI_InitStruct.SPI_NSSSlectEn = SPI_NSS_SELECT_DISABLE;
  SPI_Init(SPI0, &SPI_InitStruct);
  /* SPI Slave High Speed */
  SPI_SlaveHighSpeedCmd(SPI0,ENABLE);
}

/**
 *******************************************************************************
 * @brief   SPI GPIO Init.
 * @param   None.
 * @retval  None.
 ******************************************************************************/
void SPI_GPIOInit(void)
{
  GPIO_InitTypeStruct GPIO_InitStruct;

  /* Enable clock */
  RCC_Unlock();
  RCC_AHBPeriphClockCmd(RCC_AHB_PERIPH_GPIO, ENABLE);
  RCC_Lock();
  /* Config GPIO */
  GPIO_StructInit(&GPIO_InitStruct);
  GPIO_InitStruct.GPIO_Mode = GPIO_MODE_ALT_FUNC;
  GPIO_InitStruct.GPIO_Speed = GPIO_SPEED_HIGH;
  GPIO_InitStruct.GPIO_OType = GPIO_OTYPE_PP;
  GPIO_InitStruct.GPIO_PuPd = GPIO_PUPD_NOPULL;
  //SCLK
  GPIO_InitStruct.GPIO_Pin = GPIO_PIN_3;//PA5
  GPIO_Init(GPIOB, &GPIO_InitStruct);
  GPIO_AltFuncConfig(GPIOB, GPIO_PIN_3, GPIO_ALT_FUNC_0);
  //MISO
  GPIO_InitStruct.GPIO_Pin = GPIO_PIN_4;//PA6
  GPIO_Init(GPIOB, &GPIO_InitStruct);
  GPIO_AltFuncConfig(GPIOB, GPIO_PIN_4, GPIO_ALT_FUNC_0);
  //MOSI
  GPIO_InitStruct.GPIO_Pin = GPIO_PIN_5;//PA7
  GPIO_Init(GPIOB, &GPIO_InitStruct);
  GPIO_AltFuncConfig(GPIOB, GPIO_PIN_5, GPIO_ALT_FUNC_0);
  //CS
  GPIO_InitStruct.GPIO_Mode = GPIO_MODE_OUTPUT;
  GPIO_InitStruct.GPIO_PuPd = GPIO_PUPD_UP;
  GPIO_InitStruct.GPIO_Pin = GPIO_PIN_15;//PA4
  GPIO_Init(GPIOA, &GPIO_InitStruct);
  GPIO_AltFuncConfig(GPIOA, GPIO_PIN_15, GPIO_ALT_FUNC_0);
}
/**
 *******************************************************************************
 * @brief   SPI SlaveInterrupt Sample.
 * @param   None.
 * @retval  None.
 ******************************************************************************/
void SPI_SlaveInterrupt_Sample(void)
{
  /* GPIO Init */
  SPI_GPIOInit();
  
  /* SPI config */
  SPI_Config();
    
  /* Enable SPI0 */
  SPI_Cmd(SPI0, ENABLE);
  
  /* Clear interrupt count value */
  int_flag_cnt_txe = 0;
  int_flag_cnt_rxne = 0;
  /* Enable interrupt */
  SPI_IntCmd(SPI0, SPI_INT_TYPE_TXE, ENABLE);
  SPI_IntCmd(SPI0, SPI_INT_TYPE_RXNE, ENABLE);
  NVIC_SetPriority(SPI0_IRQn, 1);
  NVIC_EnableIRQ(SPI0_IRQn);
  DelayMs(1000);
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/************************ (C) COPYRIGHT DP *****END OF FILE ***************************/
