/**
 *******************************************************************************
 * @file    tim_cascadetrigger.c
 *
 * @brief   Implement system specific functions.
 *
 * @author  DP
 *
 * @version V1.0.0
 *
 * @date    2024-12-01
 *
 *******************************************************************************/

/*
    Copyright (c) 2024, DP Microelectronics Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

/* Files include */
#include "tim_cascadetrigger.h"

/**
  * @addtogroup DPM32M015_LibSamples
  * @{
  */

/**
  * @addtogroup TIM
  * @{
  */

/**
  * @addtogroup TIM CascadeTrigger
  * @{
  */

/* Private typedef --------------------------------------------------------------*/
TIM_TypeStruct TIM_Map[] = {
  {TIM0, TIM0_IRQn, GPIOA, GPIO_PIN_0},
  {TIM1, TIM1_IRQn, GPIOA, GPIO_PIN_1},
};

TIM_InitTypeStruct TIM_InitStruct;

/* Private define ---------------------------------------------------------------*/

/* Private macro ----------------------------------------------------------------*/

/* Private variables ------------------------------------------------------------*/

/* Private functions ------------------------------------------------------------*/
/**
 *******************************************************************************
 * @brief   TIMx IRQ Handler.
 * @param   None.
 * @retval  None.
 ******************************************************************************/
void TIMx_IRQ_Handler(uint8_t TIM_Index)
{    
  if(TIM_GetIntFlagStatus(TIM_Map[TIM_Index].port) != RESET)
  {
     TIM_ClearIntFlag(TIM_Map[TIM_Index].port);
     GPIO_ToggleBit(TIM_Map[TIM_Index].gpio, TIM_Map[TIM_Index].pin);
  }
}

/**
 *******************************************************************************
 * @brief   TIM GPIO Init.
 * @param   None.
 * @retval  None.
 ******************************************************************************/
void TIM_GPIOInit(TIM_TypeStruct* TIM_Struct)
{
  /* Enable clock */
  RCC_Unlock();
  RCC_AHBPeriphClockCmd(RCC_AHB_PERIPH_GPIO, ENABLE);
  RCC_Lock();
  
  GPIO_InitTypeStruct TIM_GPIO_InitStruct;
  GPIO_StructInit(&TIM_GPIO_InitStruct);
  TIM_GPIO_InitStruct.GPIO_Mode = GPIO_MODE_OUTPUT;
  TIM_GPIO_InitStruct.GPIO_Speed = GPIO_SPEED_HIGH;
  TIM_GPIO_InitStruct.GPIO_OType = GPIO_OTYPE_PP;
  TIM_GPIO_InitStruct.GPIO_PuPd = GPIO_PUPD_NOPULL;

  TIM_GPIO_InitStruct.GPIO_Pin = TIM_Struct->pin;
  GPIO_Init(TIM_Struct->gpio, &TIM_GPIO_InitStruct);

  GPIO_ResetBit(TIM_Struct->gpio, TIM_Struct->pin);
}

/**
 *******************************************************************************
 * @brief   Time Init.
 * @param   None.
 * @retval  None.
 ******************************************************************************/
void Time_InitConfig(void)
{
  /* Enable clock */
  RCC_Unlock();
  RCC_APBPeriphClockCmd(RCC_APB_PERIPH_TIM0, ENABLE);
  RCC_APBPeriphClockCmd(RCC_APB_PERIPH_TIM1, ENABLE);
  RCC_Lock();
  
  /* Config TIM */
  TIM_StructInit(&TIM_InitStruct);
  TIM_InitStruct.TIM_ReloadValue = 48000;
  TIM_InitStruct.TIM_CounterCycle = TIM_COUNTER_CYCLE_PERIOD;
  TIM_InitStruct.TIM_IntState = ENABLE;
  TIM_Init(TIM0, &TIM_InitStruct);
  TIM_InitStruct.TIM_ReloadValue = 48000*10;
  TIM_Init(TIM1, &TIM_InitStruct);
}

/**
 *******************************************************************************
 * @brief   Demo Sample.
 * @param   None.
 * @retval  None.
 ******************************************************************************/
void TIM_CascadeTrigger_Sample(void)
{
  /* GPIO Init */
  TIM_GPIOInit(&TIM_Map[0]);
  TIM_GPIOInit(&TIM_Map[1]);
  
  /* Time_InitConfig */
  Time_InitConfig();
  
  /* Config triggerType and triggerSource */
  TIM_TriggerTypeEnum triggerType = TIM_TRIGGER_TYPE_SOURCE_COMPLETE ;
  TIM_TriggerSourceEnum triggerSource = TIM_COUNTER_SOURCE_TIMER1;
  
  TIM_TriggerTypeConfig(TIM_Map[0].port,triggerType);
  TIM_TriggerSourceConfig(TIM_Map[0].port, triggerSource);

  /* Enable interrupt */
  NVIC_SetPriority(TIM_Map[0].irq_num, 2);
  NVIC_SetPriority(TIM_Map[1].irq_num, 2);
  NVIC_EnableIRQ(TIM_Map[0].irq_num);
  NVIC_EnableIRQ(TIM_Map[1].irq_num);  
  TIM_IntCmd(TIM_Map[0].port, ENABLE);
  TIM_IntCmd(TIM_Map[1].port, ENABLE);
  
  /* Enable TIM0 */
  TIM_Cmd(TIM_Map[0].port, ENABLE);
  /* Enable TIM1 */
  TIM_Cmd(TIM_Map[1].port, ENABLE);
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/************************ (C) COPYRIGHT DP *****END OF FILE ***************************/
