/**
 *******************************************************************************
 * @file    dpm32m0xx_dac.c.
 *
 * @brief   Source file for DAC firmware driver.
 *          This file provides firmware functions to manage the following :
 *          functionalities of the Digital-to-Analog Converter (DAC) peripheral:
 *           + Initialization functions.
 *           + Configuration functions.
 *
 * @author  DPM
 *
 * @version V1.0.0
 *
 * @date    2023-11-01
 *
 * @verbatim
 ===============================================================================
                       ##### How to use this driver #####
 ===============================================================================
    [..]
      (#) Enable The DAC controller clock using
          RCC_APBPeriphClockCmd(RCC_APB_PERIPH_DAC, ENABLE)

      (#) DAC pin configuration
        (+) Enable the clock for the DAC GPIO using the following function:
              RCC_AHBPeriphClockCmd(RCC_AHB_PERIPH_GPIO, ENABLE)
        (+) Configure these DAC pins in analog mode using GPIO_Init() function
        (+) Set these DAC pins analog alternate functionin using
            GPIO_AnaAFConfig() function

      (#) Configure the DAC using DAC_Init() function

      (#) Enable the DAC using DAC_Cmd() function

 * @endverbatim
 *******************************************************************************/

#include "dpm32m0xx_dac.h"


#if defined (DPM32M08x) || defined (DPM32M05x) || defined (DPM32M03x)

/**
 *******************************************************************************
 * @brief  Deinitializes DAC peripheral registers to their default reset values.
 * @param  None
 * @retval None
 ******************************************************************************/
void DAC_DeInit(void)
{
  /* Set control register to reset value */
  DAC->CR = 0x00000000;

  /* Set data register to reset value */
  DAC->DR = 0x00000000;
}

/**
 *******************************************************************************
 * @brief  Initializes the DAC peripheral according to the specified parameters
 *         in the DAC_InitStruct.
 * @param  [in]  DAC_InitType: Structure pointer of DAC configuration.
 * @retval None.
 ******************************************************************************/
void DAC_Init(DAC_InitTypeStruct *DAC_InitType)
{
  uint32_t tmpReg = 0UL;

  /* Parameters check. */
  PARAM_ASSERT(IS_VREF_SEL(DAC_InitType->DAC_Vref));

  /* Read the value of the DAC control register. */
  tmpReg = DAC->CR;

  /* Set DAC Vref value.  */
  tmpReg &= ((~DAC_CR_VREF_SEL_Msk));
  tmpReg |= (uint32_t)(DAC_InitType->DAC_Vref << DAC_CR_VREF_SEL_Pos);

  /* Set the value of the DAC control register.*/
  DAC->CR = tmpReg;

  /* Read the value of the DAC data register. */
  tmpReg = DAC->DR;

  /* Set DAC data register. */
  tmpReg &= ~DAC_DR_DATA_Msk;
  tmpReg |= (uint32_t)((DAC_InitType->DAC_Data & 0x0FFF) << DAC_DR_DATA_Pos);

  /* Set the value of the DAC data register.*/
  DAC->DR = tmpReg;
}

/**
 *******************************************************************************
 * @brief   Initialize the DAC_InitType with default parameters.
 * @param   [in]  DAC_InitType: Pointer to a DAC_InitTypeStruct structure which
 *                will be initialized.
 * @retval  None.
 ******************************************************************************/
void DAC_StructInit(DAC_InitTypeStruct *DAC_InitType)
{
  DAC_InitType->DAC_Vref = DAC_VREF_AVDD;
  DAC_InitType->DAC_Data = 0x00;
}

/**
 *******************************************************************************
 * @brief   Enables or disables the specified DAC peripheral.
 * @param   [in]  NewState: This parameter can be: ENABLE or DISABLE.
 * @retval  None.
 ******************************************************************************/
void DAC_Cmd(FunctionalState NewState)
{
  if(DISABLE != NewState)
  {
    /* Enable DAC peripheral. */
    DAC->CR |= DAC_CR_EN_Msk;
  }
  else
  {
    /* Disable DAC peripheral. */
    DAC->CR &= ~DAC_CR_EN_Msk;
  }
}

/**
 *******************************************************************************
 * @brief   Enable the DAC peripheral.
 * @param   None.
 * @retval  None.
 ******************************************************************************/
void DAC_Enable(void)
{
  /* Enable DAC peripheral. */
  DAC->CR |= DAC_CR_EN_Msk;
}

/**
 *******************************************************************************
 * @brief   Disable the DAC peripheral.
 * @param   None.
 * @retval  None.
 ******************************************************************************/
void DAC_Disable(void)
{
  /* Disable DAC  peripheral. */
  DAC->CR &= ~DAC_CR_EN_Msk;
}

/**
 *******************************************************************************
 * @brief  Selects the DAC Vref.
 * @param  DAC_Vref: Specifies the DAC Vref.
 *         This parameter can be a value of @ref DAC_VrefSelEnum .
 *            @arg DAC_VREF_Bandgap: DAC Vref select Bandgap 2V
 *            @arg DAC_VREF_AVDD: DAC Vref select AVDD
 ******************************************************************************/
void DAC_VrefSelect(DAC_VrefSelEnum DAC_Vref)
{
  uint32_t tmpReg = 0UL;

  /* Get CR register value */
  tmpReg = DAC->CR;

  /* Reset DAC_VREF bit */
  tmpReg &= ~((uint32_t)DAC_CR_VREF_SEL_Msk);

  /* Set the DAC_VREF bit */
  tmpReg |= (uint32_t)(DAC_Vref << DAC_CR_VREF_SEL_Pos);

  /* Write to CR register */
  DAC->CR = (uint32_t)tmpReg;
}

/**
 *******************************************************************************
 * @brief   Set the DAC Converted data.
 * @param   DAC_Data: DAC Converted data.
 * @retval  None.
 ******************************************************************************/
void DAC_SetData(uint16_t DAC_Data)
{
  /* Set the value of the DAC DR register.*/
  DAC->DR = DAC_Data & 0x0FFF;
}

#endif /* DPM32M08x || DPM32M05x || DPM32M03x */
